import { jsx, jsxs } from "react/jsx-runtime";
import {
  DEFAULT_SUPPORTED_IMAGE_TYPES,
  DEFAULT_SUPPORT_VIDEO_TYPES,
  DefaultSpinner,
  ErrorScreen,
  LoadingScreen,
  TldrawEditor,
  useEditor,
  useEditorComponents,
  useOnMount,
  useShallowArrayIdentity,
  useShallowObjectIdentity
} from "@tldraw/editor";
import { useMemo } from "react";
import { TldrawHandles } from "./canvas/TldrawHandles.mjs";
import { TldrawScribble } from "./canvas/TldrawScribble.mjs";
import { TldrawSelectionBackground } from "./canvas/TldrawSelectionBackground.mjs";
import { TldrawSelectionForeground } from "./canvas/TldrawSelectionForeground.mjs";
import { TldrawShapeIndicators } from "./canvas/TldrawShapeIndicators.mjs";
import { defaultBindingUtils } from "./defaultBindingUtils.mjs";
import {
  registerDefaultExternalContentHandlers
} from "./defaultExternalContentHandlers.mjs";
import { defaultShapeTools } from "./defaultShapeTools.mjs";
import { defaultShapeUtils } from "./defaultShapeUtils.mjs";
import { registerDefaultSideEffects } from "./defaultSideEffects.mjs";
import { defaultTools } from "./defaultTools.mjs";
import { TldrawUi } from "./ui/TldrawUi.mjs";
import { useTldrawUiComponents } from "./ui/context/components.mjs";
import { useToasts } from "./ui/context/toasts.mjs";
import { usePreloadAssets } from "./ui/hooks/usePreloadAssets.mjs";
import { useTranslation } from "./ui/hooks/useTranslation/useTranslation.mjs";
import { useDefaultEditorAssetsWithOverrides } from "./utils/static-assets/assetUrls.mjs";
function Tldraw(props) {
  const {
    children,
    maxImageDimension,
    maxAssetSize,
    acceptedImageMimeTypes,
    acceptedVideoMimeTypes,
    onMount,
    components = {},
    shapeUtils = [],
    bindingUtils = [],
    tools = [],
    ...rest
  } = props;
  const _components = useShallowObjectIdentity(components);
  const componentsWithDefault = useMemo(
    () => ({
      Scribble: TldrawScribble,
      ShapeIndicators: TldrawShapeIndicators,
      CollaboratorScribble: TldrawScribble,
      SelectionForeground: TldrawSelectionForeground,
      SelectionBackground: TldrawSelectionBackground,
      Handles: TldrawHandles,
      ..._components
    }),
    [_components]
  );
  const _shapeUtils = useShallowArrayIdentity(shapeUtils);
  const shapeUtilsWithDefaults = useMemo(
    () => [...defaultShapeUtils, ..._shapeUtils],
    [_shapeUtils]
  );
  const _bindingUtils = useShallowArrayIdentity(bindingUtils);
  const bindingUtilsWithDefaults = useMemo(
    () => [...defaultBindingUtils, ..._bindingUtils],
    [_bindingUtils]
  );
  const _tools = useShallowArrayIdentity(tools);
  const toolsWithDefaults = useMemo(
    () => [...defaultTools, ...defaultShapeTools, ..._tools],
    [_tools]
  );
  const assets = useDefaultEditorAssetsWithOverrides(rest.assetUrls);
  const { done: preloadingComplete, error: preloadingError } = usePreloadAssets(assets);
  if (preloadingError) {
    return /* @__PURE__ */ jsx(ErrorScreen, { children: "Could not load assets. Please refresh the page." });
  }
  if (!preloadingComplete) {
    return /* @__PURE__ */ jsx(LoadingScreen, { children: /* @__PURE__ */ jsx(DefaultSpinner, {}) });
  }
  return /* @__PURE__ */ jsx(
    TldrawEditor,
    {
      initialState: "select",
      ...rest,
      components: componentsWithDefault,
      shapeUtils: shapeUtilsWithDefaults,
      bindingUtils: bindingUtilsWithDefaults,
      tools: toolsWithDefaults,
      children: /* @__PURE__ */ jsxs(TldrawUi, { ...rest, components: componentsWithDefault, children: [
        /* @__PURE__ */ jsx(
          InsideOfEditorAndUiContext,
          {
            maxImageDimension,
            maxAssetSize,
            acceptedImageMimeTypes,
            acceptedVideoMimeTypes,
            onMount
          }
        ),
        children
      ] })
    }
  );
}
function InsideOfEditorAndUiContext({
  maxImageDimension = 5e3,
  maxAssetSize = 10 * 1024 * 1024,
  // 10mb
  acceptedImageMimeTypes = DEFAULT_SUPPORTED_IMAGE_TYPES,
  acceptedVideoMimeTypes = DEFAULT_SUPPORT_VIDEO_TYPES,
  onMount
}) {
  const editor = useEditor();
  const toasts = useToasts();
  const msg = useTranslation();
  useOnMount(() => {
    const unsubs = [];
    unsubs.push(registerDefaultSideEffects(editor));
    registerDefaultExternalContentHandlers(
      editor,
      {
        maxImageDimension,
        maxAssetSize,
        acceptedImageMimeTypes,
        acceptedVideoMimeTypes
      },
      {
        toasts,
        msg
      }
    );
    unsubs.push(editor.store.props.onMount(editor));
    unsubs.push(onMount?.(editor));
    return () => {
      unsubs.forEach((fn) => fn?.());
    };
  });
  const { Canvas } = useEditorComponents();
  const { ContextMenu } = useTldrawUiComponents();
  if (ContextMenu) {
    return /* @__PURE__ */ jsx(ContextMenu, {});
  }
  if (Canvas) {
    return /* @__PURE__ */ jsx(Canvas, {});
  }
  return null;
}
export {
  Tldraw
};
//# sourceMappingURL=Tldraw.mjs.map
